/* validation/validazione_sezione_allegati_liquidazione.js */
(function () {
  'use strict';

  // --- utility grafiche (solo Bootstrap) ---
  // --- utility grafiche (uniforme alle altre sezioni) ---
  function paintHeader(errorCount) {
    const headerBtn = document.querySelector('#allegati_liquidazione .accordion-button');
    const badgeSpan = document.getElementById('validazione_sezione_allegati_liquidazione');
    if (!headerBtn || !badgeSpan) return;

    // reset
    headerBtn.classList.remove(
      'bg-danger', 'bg-success', 'text-white',
      'background-color-error-accordion', 'background-color-success-accordion'
    );
    badgeSpan.classList.remove('badge', 'text-bg-error', 'text-bg-success');
    badgeSpan.classList.add('badge');

    if (errorCount > 0) {
      headerBtn.classList.add('background-color-error-accordion', 'text-white');
      badgeSpan.classList.add('text-bg-error');
      badgeSpan.textContent = `Vi sono ${errorCount} errori`;
    } else {
      headerBtn.classList.add('background-color-success-accordion', 'text-white');
      badgeSpan.classList.add('text-bg-success');
      badgeSpan.textContent = 'Sezione Compilata Correttamente';
    }
  }
  function ok(el, box) { if (!el) return; el.classList.remove('is-invalid'); el.classList.add('is-valid'); if (box) box.textContent = ''; }
  function ko(el, box, msg) { if (!el) return; el.classList.add('is-invalid'); el.classList.remove('is-valid'); if (box) box.innerHTML = `<small class="text-danger">${msg}</small>`; }

  // --- validatori base ---
  const URL_RE = /^(https?:\/\/)[^\s]+$/i;
  const IMG_EXT = ['jpg', 'jpeg', 'png', 'webp', 'gif', 'bmp', 'tiff'];

  function isUrl(v) { return URL_RE.test((v || '').trim()); }
  function getFiles(input) { return (input && input.files) ? Array.from(input.files) : []; }
  function hasAllowedImages(files) {
    return files.every(f => {
      const ext = (f.name.split('.').pop() || '').toLowerCase();
      return IMG_EXT.includes(ext);
    });
  }

  // se il campo è required, verifica che ci sia almeno 1 file.
  // se l'input è multiple, verifica anche maxCount (se passato).
  function validateFileRequired(input, errBox, msgObbl, opts = {}) {
    const files = getFiles(input);
    if (!files.length) { ko(input, errBox, msgObbl); return 1; }
    if (opts.maxCount && files.length > opts.maxCount) { ko(input, errBox, `Numero massimo di file: ${opts.maxCount}.`); return 1; }
    if (opts.onlyImages && !hasAllowedImages(files)) { ko(input, errBox, 'Sono ammessi solo file immagine.'); return 1; }
    ok(input, errBox); return 0;
  }

  function validaSezioneAllegatiLiquidazione() {
    const section = document.getElementById('allegati_liquidazione');
    if (!section) return true;

    const inRelazione = document.getElementById('allegato_relazione');
    const inPubblicita = document.getElementById('allegato_documentazione_pubblicitaria');
    const inNoContr = document.getElementById('allegato_dichiarazione_no_contributi');
    const inScost = document.getElementById('allegato_scostamenti'); // nel markup è required
    const inFoto = document.getElementById('allegato_foto');        // “max 10 immagini”
    const inLinkVideo = document.getElementById('allegato_link_video');
    const eRelazione = document.getElementById('errore_allegato_relazione');
    const ePubblicita = document.getElementById('errore_allegato_documentazione_pubblicitaria');
    const eNoContr = document.getElementById('errore_allegato_dichiarazione_no_contributi');
    const eScost = document.getElementById('errore_allegato_scostamenti');
    const eFoto = document.getElementById('errore_allegato_foto');
    const eLinkVideo = document.getElementById('errore_allegato_link_video');

    let errors = 0;

    // Relazione (obbligatoria)
    errors += validateFileRequired(inRelazione, eRelazione, 'Relazione finale obbligatoria.');

    // Documentazione multimediale/pubblicitaria (obbligatoria, qualsiasi formato)
    errors += validateFileRequired(inPubblicita, ePubblicita, 'Documentazione multimediale/pubblicitaria obbligatoria.');

    // Dichiarazione no contributi (obbligatoria)
    errors += validateFileRequired(inNoContr, eNoContr, 'Dichiarazione sostitutiva obbligatoria.');

    // Scostamenti (il tuo input è `required`: lo tratto come obbligatorio)
    // Scostamenti (OPZIONALE: valida solo se compilato)
    if (inScost && inScost.files.length > 0) {
      errors += validateFileRequired(inScost, eScost, 'Motivazione scostamenti obbligatoria.');
    } else {
      ok(inScost, eScost); // se vuoto non è errore
    }
    // Foto (obbligatorie, solo immagini, max 10 se l’input è multiple)
    const fotoOpts = { onlyImages: true };
    if (inFoto?.hasAttribute('multiple')) fotoOpts.maxCount = 10;
    errors += validateFileRequired(inFoto, eFoto, 'Documentazione fotografica obbligatoria.', fotoOpts);

    // Link video (obbligatorio e URL valido)
    // Link video (OPZIONALE: valida solo se compilato)
    if (inLinkVideo) {
      const linkVal = (inLinkVideo.value || '').trim();
      if (linkVal && !isUrl(linkVal)) {
        errors++; ko(inLinkVideo, eLinkVideo, 'URL non valido. Inserisci http:// o https://');
      } else {
        // vuoto (ok) o valido
        inLinkVideo.classList.remove('is-invalid');
        if (linkVal) inLinkVideo.classList.add('is-valid'); else inLinkVideo.classList.remove('is-valid');
        if (eLinkVideo) eLinkVideo.textContent = '';
      }
    }

    // Aggiorna header e badge
    paintHeader(errors);

    // Se errori: apri accordion e scrolla alla prima anomalia
    if (errors > 0) {
      const collapse = document.getElementById('collapseEight');
      if (collapse && !collapse.classList.contains('show') && window.bootstrap?.Collapse) {
        new bootstrap.Collapse(collapse, { toggle: true });
      }
      const firstInvalid = section.querySelector('.is-invalid');
      (firstInvalid || section).scrollIntoView({ behavior: 'smooth', block: 'start' });
      return false;
    }
    return true;
  }

  function init() {
    const badge = document.getElementById('validazione_sezione_allegati_liquidazione');
    if (badge) badge.innerHTML = '';
    // valida SOLO al click su "Crea Files"
    document.querySelectorAll('.crea-files').forEach(btn => {
      btn.addEventListener('click', (e) => {
        const ok = validaSezioneAllegatiLiquidazione();
        if (!ok) e.preventDefault();
      });
    });
  }

  // Espongo se serve altrove
  window.ValidationAllegatiLiquidazione = {
    validate: validaSezioneAllegatiLiquidazione,
    init
  };

  document.addEventListener('DOMContentLoaded', init);
})();