/* validation/validazione_sezione_dati_bancari.js */
(function () {
  'use strict';

  /* -------------------------
     Helper IBAN
  -------------------------- */

  // IT + 2 cifre + 1 lettera + 10 cifre + 12 alfanumerici (27 caratteri totali)
  var regIbanIT = /^IT\d{2}[A-Z]\d{10}[0-9A-Z]{12}$/;

  function normalizeIban(s) {
    return (s || '').toString().replace(/\s+/g, '').toUpperCase();
  }

  // Controllo modulo 97 (standard IBAN)
  function ibanModulo97Ok(iban) {
    var clean = normalizeIban(iban);
    if (clean.length < 4) return false;

    // sposto le prime 4 in fondo
    var rearranged = clean.slice(4) + clean.slice(0, 4);
    var remainder = 0;

    for (var i = 0; i < rearranged.length; i++) {
      var ch = rearranged.charAt(i);
      var code = rearranged.charCodeAt(i);

      var chunk;
      if (code >= 65 && code <= 90) {          // A-Z
        chunk = String(code - 55);             // A->10 ... Z->35
      } else if (code >= 48 && code <= 57) {   // 0-9
        chunk = ch;
      } else {
        return false;
      }

      for (var j = 0; j < chunk.length; j++) {
        remainder = (remainder * 10 + (chunk.charCodeAt(j) - 48)) % 97;
      }
    }
    return remainder === 1;
  }

  /* -------------------------
     Grafica header / badge
  -------------------------- */

  function paintHeaderDatiBancari(errorCount) {
    const headerBtn = document.querySelector('#dati_bancari .accordion-button');
    const badgeSpan = document.getElementById('validazione_sezione_dati_bancari');
    if (!headerBtn || !badgeSpan) return;

    // reset classi
    headerBtn.classList.remove(
      'bg-danger', 'bg-success', 'text-white',
      'background-color-error-accordion',
      'background-color-success-accordion'
    );
    badgeSpan.className = ''; // tolgo tutte le classi
    badgeSpan.textContent = '';

    if (errorCount > 0) {
      headerBtn.classList.add('background-color-error-accordion', 'text-white');
      badgeSpan.classList.add('badge', 'text-bg-error');
      const testo = 'Vi ' +
        (errorCount === 1 ? 'è ' : 'sono ') +
        errorCount + ' ' +
        (errorCount === 1 ? 'errore' : 'errori');
      badgeSpan.textContent = testo;
    } else {
      headerBtn.classList.add('background-color-success-accordion', 'text-white');
      badgeSpan.classList.add('badge', 'text-bg-success');
      badgeSpan.textContent = 'Sezione Compilata Correttamente';
    }
  }

  /* -------------------------
     Validazione centrale
     options = { updateHeader: true/false }
  -------------------------- */

  function validateDatiBancari(options) {
    var updateHeader = !options || options.updateHeader !== false; // default: true
    var errors = 0;

    var ibanEl = document.getElementById('iban');
    var istEl  = document.getElementById('istituto-credito');
    var agEl   = document.getElementById('agenzia_banca');

    var errIban = document.getElementById('errore_iban');
    var errIst  = document.getElementById('errore_istituto_credito');
    var errAg   = document.getElementById('errore_agenzia_banca');

    // ===== IBAN =====
    if (ibanEl) {
      var iban = normalizeIban(ibanEl.value || '');
      ibanEl.value = iban;

      if (!iban) {
        errors++;
        ibanEl.classList.add('is-invalid');
        ibanEl.classList.remove('is-valid');
        if (errIban) {
          errIban.classList.add('invalid-feedback');
          errIban.textContent = 'IBAN obbligatorio.';
        }
      } else if (iban.startsWith('IT') && iban.length !== 27) {
        errors++;
        ibanEl.classList.add('is-invalid');
        ibanEl.classList.remove('is-valid');
        if (errIban) {
          errIban.classList.add('invalid-feedback');
          errIban.textContent = 'L’IBAN italiano deve contenere 27 caratteri.';
        }
      } else if (!regIbanIT.test(iban) || !ibanModulo97Ok(iban)) {
        errors++;
        ibanEl.classList.add('is-invalid');
        ibanEl.classList.remove('is-valid');
        if (errIban) {
          errIban.classList.add('invalid-feedback');
          errIban.textContent = 'IBAN non valido.';
        }
      } else {
        ibanEl.classList.remove('is-invalid');
        ibanEl.classList.add('is-valid');
        if (errIban) {
          errIban.classList.remove('invalid-feedback');
          errIban.textContent = '';
        }
      }
    }

    // ===== Istituto di Credito =====
    if (istEl) {
      var ist = (istEl.value || '').trim();
      if (!ist) {
        errors++;
        istEl.classList.add('is-invalid');
        istEl.classList.remove('is-valid');
        if (errIst) {
          errIst.classList.add('invalid-feedback');
          errIst.textContent = 'Istituto di credito obbligatorio.';
        }
      } else {
        istEl.classList.remove('is-invalid');
        istEl.classList.add('is-valid');
        if (errIst) {
          errIst.classList.remove('invalid-feedback');
          errIst.textContent = '';
        }
      }
    }

    // ===== Agenzia =====
    if (agEl) {
      var ag = (agEl.value || '').trim();
      if (!ag) {
        errors++;
        agEl.classList.add('is-invalid');
        agEl.classList.remove('is-valid');
        if (errAg) {
          errAg.classList.add('invalid-feedback');
          errAg.textContent = 'Agenzia obbligatoria.';
        }
      } else {
        agEl.classList.remove('is-invalid');
        agEl.classList.add('is-valid');
        if (errAg) {
          errAg.classList.remove('invalid-feedback');
          errAg.textContent = '';
        }
      }
    }

    // 👉 colora l’header SOLO se updateHeader = true
    if (updateHeader) {
      paintHeaderDatiBancari(errors);
    }

    return errors;
  }

  /* -------------------------
     Validazioni live
     (NON toccano l’header)
  -------------------------- */

  function initLive() {
    $(document).on('input blur', '#iban, #istituto-credito, #agenzia_banca', function () {
      // live: aggiorno solo i campi, NON l’header
      validateDatiBancari({ updateHeader: false });
    });
  }

  document.addEventListener('DOMContentLoaded', initLive);

  // Espongo la funzione che userà crea_files.js
  window.validateSezioneDatiBancari = function () {
    // qui vogliamo anche l’header aggiornato
    return validateDatiBancari({ updateHeader: true });
  };
})();