// =======================
//  REGOLE & HELPER
// =======================
const REGEX_STRING      = /^.{2,}$/;
const REGEX_DATA        = /^\d{4}\-(0[1-9]|1[0-2])\-(0[1-9]|[12][0-9]|3[01])$/;
const REGEX_MONEY_DOT   = /^\d+(\.\d{1,2})?$/;  // numeri con . decimale
const REGEX_IMPORTO_ANY = /^\d+([.,]\d{1,2})?$/;

// Se NON lo hai già definito in un altro file, puoi lasciare questo.
// Se esiste già altrove, commentalo per non avere doppie dichiarazioni.
const VALIDAZIONI_ALLEGATI = {
  comuni: {
    delibera_affidamento: {
      msg: "Delibera di affidamento obbligatoria.",
      errore: ".errore_delibera_affidamento"
    },
    delibera_pagamento: {
      msg: "Delibera di pagamento obbligatoria.",
      errore: ".errore_delibera_pagamento"
    },
    mandato_pagamento: {
      msg: "Mandato di pagamento obbligatorio.",
      errore: ".errore_mandato_pagamento"
    },
    fattura: {
      msg: "Documento fattura/titolo obbligatorio.",
      errore: ".errore_fattura"
    }
  },
  associazioni: {
    fattura: {
      msg: "Documento fattura/titolo obbligatorio.",
      errore: ".errore_fattura"
    },
    ricevuta_bonifico_assegno: {
      msg: "Ricevuta di bonifico/assegno obbligatoria.",
      errore: ".errore_ricevuta_bonifico_assegno"
    }
  },
  opt: {
    prospetto_riparto: {
      msg: "Prospetto di riparto obbligatorio.",
      errore: ".errore_prospetto_riporto"
    }
  }
};

function getTipoBeneficiario() {
  const ben = $('input[name=tipo_beneficiario]:checked').val() || '';
  // ben è 'a', 'b', 'c', 'd', 'e', 'f', 'g'

  if (!ben) return '';

  // A, B, C -> COMUNI PUBBLICI
  if (['a', 'b', 'c'].includes(ben)) {
    return 'comuni';
  }

  // D, E, F, G -> ASSOCIAZIONI / PRIVATI
  if (['d', 'e', 'f', 'g'].includes(ben)) {
    return 'associazioni';
  }

  return '';
}
// helper grafico base
function setCampoNonValido($input, $err, msg) {
  $input.addClass('is-invalid').removeClass('is-valid');
  if ($err && $err.length) {
    $err.addClass('invalid-feedback')
        .html(`<small class="text-danger">${msg}</small>`);
  }
}

function setCampoValido($input, $err) {
  $input.removeClass('is-invalid').addClass('is-valid');
  if ($err && $err.length) {
    $err.removeClass('invalid-feedback').html('');
  }
}

// Validazione base LIVE (non conta errori, solo stile)
function validaCampoBase($input, erroreSelector, validator, msgFormato, msgObblig) {
  let $err;

  if (erroreSelector.startsWith('#')) {
    $err = $(erroreSelector);
  } else {
    // negli investimenti l'errore è sempre subito dopo l'input
    $err = $input.siblings(erroreSelector);
  }

  const val = ($input.val() || '').trim();

  if (!val) {
    setCampoNonValido($input, $err, msgObblig);
    return false;
  }

  let ok = true;
  if (validator) {
    if (validator instanceof RegExp) ok = validator.test(val);
    else if (typeof validator === 'function') ok = !!validator(val);
  }

  if (!ok) {
    setCampoNonValido($input, $err, msgFormato);
    return false;
  }

  setCampoValido($input, $err);
  return true;
}

// =======================
//  VALIDAZIONI LIVE
// =======================

// --- PRIMA TABELLA: INVESTIMENTI ---

// Oggetto della spesa
$(document).on("input", "input[name=oggetto_spesa]", function () {
  const $input = $(this);

  // 1) valido il campo
  const isValid = validaCampoBase(
    $input,
    '.errore_oggetto_spesa',
    REGEX_STRING,
    "Il valore inserito deve avere almeno due caratteri.",
    "Oggetto della spesa obbligatorio."
  );

  // 2) se NON è valido, non faccio nulla sugli allegati
  if (!isValid) return;

  // 3) recupero la riga (id tipo "oggetto_spesa_1")
  const $tr = $input.closest('tr');
  const id_riga = $tr.attr("id"); // es. "oggetto_spesa_1"
  if (!id_riga) return;

  // 4) estraggo l'indice numerico (es. "1")
  let index = null;
  if (id_riga.indexOf("oggetto_spesa_") === 0) {
    index = id_riga.substring("oggetto_spesa_".length);
  }
  if (!index) return;
  console.log("ds",index,$input.val());
  // 5) se esiste la funzione globale che crea la riga negli allegati, la chiamo
  if (typeof crea_riga_allegati === "function") {
    crea_riga_allegati(index, {
      oggetto: $input.val()
    });
  }
});

// Data della spesa
$(document).on("input", "input[name=data_spesa]", function () {
  validaCampoBase(
    $(this),
    '.errore_data_spesa',
    REGEX_DATA,
    "Data non valida (AAAA-MM-GG).",
    "Data della spesa obbligatoria."
  );
});

// Beneficiario
$(document).on("input", "input[name=beneficiario_spesa]", function () {
  validaCampoBase(
    $(this),
    '.errore_beneficiario_spesa',
    REGEX_STRING,
    "Il valore inserito deve avere almeno due caratteri.",
    "Beneficiario della spesa obbligatorio."
  );
});

// CUAA live (maiuscolo e formato base)
$(document).on('input blur', 'input[name=cuaa_spesa]', function () {
  const $input = $(this);
  const raw = $input.val() || '';
  const upper = raw.toUpperCase().replace(/[^A-Z0-9]/g, '');
  if (upper !== raw) $input.val(upper);

  const $err = $input.siblings('.errore_cuaa_spesa');
  const val = upper.trim();

  if (!val) {
    setCampoNonValido($input, $err, "Codice CUAA obbligatorio.");
    return;
  }

  // se hai una funzione globale controllaCFPIVA, puoi usarla qui
  if (typeof controllaCFPIVA === 'function' && !controllaCFPIVA(val)) {
    setCampoNonValido($input, $err, "Codice CUAA errato.");
    return;
  }

  setCampoValido($input, $err);
});

// Importo spesa
$(document).on("input", "input[name=importo_spesa]", function () {
  const $input = $(this);
  const v = ($input.val() || '').trim();
  const $err = $input.siblings('.errore_importo_spesa');

  if (!v) {
    setCampoNonValido($input, $err, "Inserisci l'importo.");
    return;
  }
  if (v.includes(',')) {
    setCampoNonValido($input, $err, "Errore: inserisci numeri con il punto (es. 1234.56).");
    return;
  }
  if (!REGEX_MONEY_DOT.test(v)) {
    setCampoNonValido($input, $err, "Importo non valido. Formato ammesso: 1234 o 1234.56.");
    return;
  }

  setCampoValido($input, $err);

  // se hai una funzione calcola_totale(), richiamala:
  if (typeof calcola_totale === 'function') {
    calcola_totale();
  }
});

// Numero fattura / Titolo
$(document).on("input", "input[name=n_fattura_spesa]", function () {
  validaCampoBase(
    $(this),
    '.errore_n_fattura_spesa',
    REGEX_STRING,
    "Il valore inserito deve avere almeno due caratteri.",
    "Identificativo della fattura o del titolo obbligatorio."
  );
});

// --- SECONDA TABELLA: ALLEGATI ---
$(document).on(
  'change',
  'input[name=delibera_affidamento], input[name=delibera_pagamento], input[name=mandato_pagamento], input[name=fattura], input[name=ricevuta_bonifico_assegno], input[name=prospetto_riparto]',
  function () {
    const nome_campo = $(this).attr('name');
    const tipo = getTipoBeneficiario();
    const cfg = (VALIDAZIONI_ALLEGATI[tipo]?.[nome_campo]) || (VALIDAZIONI_ALLEGATI.opt?.[nome_campo]);
    if (!cfg) return;

    const $input = $(this);
    const $err = $input.siblings(cfg.errore);
    const file = $input[0].files[0];

    if (!file) {
      setCampoNonValido($input, $err, cfg.msg);
    } else {
      setCampoValido($input, $err);
    }
  }
);

// --- DRS LIVE ---

$(document).on('input blur', '#numeroDRS', function () {
  validaCampoBase(
    $(this),
    '#errore_numeroDRS',
    REGEX_STRING,
    "Numero DRS non valido.",
    "Numero DRS obbligatorio."
  );
});

$(document).on('input blur', '#dataDRS', function () {
  const $input = $(this);
  const $err = $('#errore_dataDRS');
  const val = ($input.val() || '').trim();

  if (!val) {
    setCampoNonValido($input, $err, "Data del DRS obbligatoria.");
    return;
  }

  // type="date" garantisce già il formato
  setCampoValido($input, $err);
});

$(document).on('input blur', '#importoAmmissibile', function () {
  const $input = $(this);
  const $err = $('#errore_importoAmmissibile');
  let v = ($input.val() || '').toString().trim().replace(',', '.');

  if (!v) {
    setCampoNonValido($input, $err, "Importo DRS obbligatorio.");
    return;
  }
  if (!REGEX_MONEY_DOT.test(v)) {
    setCampoNonValido($input, $err, "Importo non valido. Formato ammesso: 1234 o 1234.56.");
    return;
  }

  setCampoValido($input, $err);
  updateAlertImportoDrsLive(); // aggiorna messaggio live se serve
});

// LIVE: solo testo dell’alert DRS vs totale (non blocca, non conta errori)
function updateAlertImportoDrsLive() {
  const $alert = $('#alert_importo_drs');

  let impDecreto = ($('#importoAmmissibile').val() || '').toString().trim().replace(',', '.');
  const valDecreto = parseFloat(impDecreto);

  let totStr = $('#importo_totale').attr('valore') || $('#importo_totale').text() || '0';
  totStr = totStr.toString().trim().replace(',', '.');
  const valTotale = parseFloat(totStr);

  if (isNaN(valDecreto) || isNaN(valTotale) || valDecreto <= 0 || valTotale <= 0) {
    $alert.addClass('d-none').text('');
    return;
  }

  if (valDecreto <= valTotale) {
    $alert
      .removeClass('d-none')
      .text(
        `Attenzione: l'importo totale (${valTotale.toFixed(2)} €) ` +
        `non può superare il valore del decreto (${valDecreto.toFixed(2)}) €.`
      );
  } else {
    $alert.addClass('d-none').text('');
  }
}

$(document).on('input change', '#importoAmmissibile', updateAlertImportoDrsLive);
$(document).on('DOMSubtreeModified input change', '#importo_totale', updateAlertImportoDrsLive);

$(document).on('input change', '#importo_totale', updateAlertImportoDrsLive);